const express = require('express');
const fs = require('fs');
const path = require('path');
const marked = require('marked');

const app = express();
const PORT = 3000;

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

app.use(express.static(path.join(__dirname, 'public')));

marked.setOptions({
  breaks: true,
  gfm: true,
});

const POSTS_PER_PAGE = 5;

function loadPosts() {
  try {
    const data = fs.readFileSync(path.join(__dirname, 'post.json'), 'utf-8');
    const posts = JSON.parse(data);
    posts.sort((a, b) => new Date(b.date) - new Date(a.date));
    return posts;
  } catch (error) {
    console.error('Error loading posts:', error);
    return [];
  }
}

app.get('/', (req, res) => {
  const page = parseInt(req.query.page) || 1;
  const posts = loadPosts();

  const totalPages = Math.ceil(posts.length / POSTS_PER_PAGE);
  const start = (page - 1) * POSTS_PER_PAGE;
  const end = start + POSTS_PER_PAGE;
  const pagePosts = posts.slice(start, end);

  const postsPreview = pagePosts.map(post => {
    const thumbnailMatch = post.content.match(/THUMBNAIL: !\[.*?\]\((.*?)\)/);
    const thumbnailUrl = thumbnailMatch ? thumbnailMatch[1] : null;

    const plainText = post.content
      .replace(/THUMBNAIL: !\[.*?\]\(.*?\)/, '')
      .replace(/!\[.*?\]\(.*?\)/g, '')
      .replace(/<.*?>/g, '')
      .replace(/[#*`~]/g, '')
      .replace(/\n/g, ' ');
    const excerpt = plainText.length > 150 ? plainText.substring(0, 150) + '...' : plainText;

    return {
      ...post,
      excerpt: excerpt.trim(),
      thumbnailUrl: thumbnailUrl,
    };
  });

  res.render('index', {
    posts: postsPreview,
    currentPage: page,
    totalPages,
    host: req.headers.host || `localhost:${PORT}`,
  });
});

app.get('/post/:id', (req, res, next) => {
  const allPosts = loadPosts();
  const post = allPosts.find(p => p.id === parseInt(req.params.id));

  if (!post) {
    return next();
  }

  const cleanContent = post.content.replace(/THUMBNAIL: !\[.*?\]\(.*?\)/, '');
  const htmlContent = marked.parse(cleanContent);
  
  const otherPosts = allPosts
    .filter(p => p.id !== post.id)
    .sort(() => 0.5 - Math.random())
    .slice(0, 3)
    .map(otherPost => {
      const thumbnailMatch = otherPost.content.match(/THUMBNAIL: !\[.*?\]\((.*?)\)/);
      const thumbnailUrl = thumbnailMatch ? thumbnailMatch[1] : null;
      
      const plainText = otherPost.content
        .replace(/THUMBNAIL: !\[.*?\]\(.*?\)/, '')
        .replace(/!\[.*?\]\(.*?\)/g, '')
        .replace(/<.*?>/g, '')
        .replace(/[#*`~]/g, '')
        .replace(/\n/g, ' ');
      const excerpt = plainText.length > 150 ? plainText.substring(0, 150) + '...' : plainText;
      
      return {
        ...otherPost,
        excerpt: excerpt.trim(),
        thumbnailUrl: thumbnailUrl,
      };
    });

  res.render('post', {
    post: {
      ...post,
      htmlContent,
    },
    otherPosts: otherPosts,
    host: req.headers.host || `localhost:${PORT}`,
  });
});

app.get('/about', (req, res) => {
  res.render('about', { title: 'Tentang Kami - Blog Minimalis' });
});

app.use((req, res, next) => {
  res.status(404).render('404', { title: 'Halaman Tidak Ditemukan' });
});

app.use((err, req, res, next) => {
  console.error(err.stack);
  res.status(500).render('500', { title: 'Terjadi Kesalahan Server' });
});

app.listen(PORT, () => {
  console.log(`Blog berjalan di http://localhost:${PORT}`);
});