const express = require('express');
const fs = require('fs');
const path = require('path');
const { marked } = require('marked');
const createDOMPurify = require('dompurify');
const { JSDOM } = require('jsdom');

// Konfigurasi sanitasi HTML
const window = new JSDOM('').window;
const DOMPurify = createDOMPurify(window);

const app = express();
const port = process.env.PORT || 3000;

// Middleware untuk file statis
app.use(express.static(path.join(__dirname, 'public')));

// View engine EJS
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// Fungsi ambil semua postingan dari JSON
function getPosts() {
    const rawData = fs.readFileSync(path.join(__dirname, 'posts.json'));
    return JSON.parse(rawData);
}

// Route 1: Beranda dengan pagination
app.get('/', (req, res) => {
    const allPosts = getPosts().sort((a, b) => new Date(b.date) - new Date(a.date));
    const page = parseInt(req.query.page) || 1;
    const postsPerPage = 5;
    const totalPosts = allPosts.length;
    const totalPages = Math.ceil(totalPosts / postsPerPage);

    const startIndex = (page - 1) * postsPerPage;
    const endIndex = startIndex + postsPerPage;
    const paginatedPosts = allPosts.slice(startIndex, endIndex);

    res.render('index', {
        posts: paginatedPosts,
        title: 'Beranda',
        totalPages,
        currentPage: page
    });
});

// Route 2: Halaman post dengan share link otomatis
app.get('/post/:slug', (req, res) => {
    const posts = getPosts();
    const post = posts.find(p => p.slug === req.params.slug);

    if (post) {
        const unsafeHtml = marked.parse(post.content);
        const safeHtml = DOMPurify.sanitize(unsafeHtml);

        const baseUrl = `${req.protocol}://${req.get('host')}`;
        const postUrl = `${baseUrl}/post/${post.slug}`;

        const shareLinks = {
            twitter: `https://twitter.com/intent/tweet?url=${encodeURIComponent(postUrl)}&text=${encodeURIComponent(post.title)}`,
            facebook: `https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(postUrl)}`,
            whatsapp: `https://api.whatsapp.com/send?text=${encodeURIComponent(post.title + ' ' + postUrl)}`
        };

        res.render('post', {
            post,
            content: safeHtml,
            postUrl,
            shareLinks
        });
    } else {
        res.status(404).send('Postingan tidak ditemukan!');
    }
});

// Route 3: Sitemap XML otomatis
app.get('/sitemap.xml', (req, res) => {
    const posts = getPosts();
    const baseUrl = `${req.protocol}://${req.get('host')}`;

    let sitemap = '<?xml version="1.0" encoding="UTF-8"?><urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
    sitemap += `<url><loc>${baseUrl}/</loc></url>`;
    posts.forEach(post => {
        sitemap += `<url><loc>${baseUrl}/post/${post.slug}</loc></url>`;
    });
    sitemap += '</urlset>';

    res.header('Content-Type', 'application/xml');
    res.send(sitemap);
});

// Jalankan server
app.listen(port, () => {
    console.log(`🚀 Server jalan di http://localhost:${port}`);
});