const express = require('express');
const path = require('path');
const session = require('express-session');
const fs = require('fs');
const app = express();
const port = 3000;


app.use(session({
    secret: 'webkelasAP-secret-key',
    resave: false,
    saveUninitialized: true,
    cookie: { maxAge: 60 * 60 * 1000 }
}));
app.use(express.urlencoded({ extended: true }));
app.use(express.static('public'));
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));


const readData = () => {
    try {
        const data = fs.readFileSync(path.join(__dirname, 'data.json'), 'utf8');
        return JSON.parse(data);
    } catch (error) {
        console.error("Error reading data.json:", error);
        return {
            namaKelas: "Kelas Tidak Ditemukan",
            waliKelas: "N/A",
            jumlahSiswaLaki: 0,
            jumlahSiswaPerempuan: 0,
            strukturKelas: {},
            pengumuman: [],
            transaksiKas: [],
            galeriFoto: [],
            jadwal: { piket: {}, pelajaran: {} }
        };
    }
};

const getScheduleForDay = (data) => {
    const days = ['minggu', 'senin', 'selasa', 'rabu', 'kamis', 'jumat', 'sabtu'];
    const today = new Date().getDay();
    const todayName = days[today];

    return {
        jadwalPiket: data.jadwal.piket[todayName] || [],
        jadwalPelajaran: data.jadwal.pelajaran[todayName] || []
    };
};


const checkAuth = (req, res, next) => {
    if (req.session.user) {
        next();
    } else {
        res.redirect('/login');
    }
};

app.get('/', checkAuth, (req, res) => {
    const dataKelas = readData();
    const todaySchedule = getScheduleForDay(dataKelas);
    
    let totalKas = 0;
    dataKelas.transaksiKas.forEach(t => {
        totalKas += (t.tipe === 'masuk' ? t.jumlah : -t.jumlah);
    });
    
    const totalSiswa = dataKelas.jumlahSiswaLaki + dataKelas.jumlahSiswaPerempuan;

    res.render('index', { 
        data: dataKelas, 
        totalKas: totalKas, 
        user: req.session.user,
        totalSiswa: totalSiswa,
        jadwalPiketHariIni: todaySchedule.jadwalPiket,
        jadwalPelajaranHariIni: todaySchedule.jadwalPelajaran
    });
});

app.get('/login', (req, res) => {
    res.render('login', { error: null });
});

app.post('/login', (req, res) => {
    const { nama } = req.body;
    if (nama && nama.trim() !== '') {
        req.session.user = nama.trim();
        res.redirect('/');
    } else {
        res.render('login', { error: 'Nama tidak boleh kosong!' });
    }
});

app.get('/logout', (req, res) => {
    req.session.destroy(err => {
        if (err) {
            return res.redirect('/');
        }
        res.clearCookie('connect.sid');
        res.redirect('/login');
    });
});

app.listen(port, () => {
    console.log(`Server kelas berjalan di http://localhost:${port}`);
});